import type { Metadata } from "next";
import { setRequestLocale } from "next-intl/server";
import { notFound } from "next/navigation";
import Image from "next/image";
import { Navbar } from "@/components/navbar";
import { Footer } from "@/modules/landing/components";
import {
  getBlogBySlug,
  getRecentBlogs,
  getTags,
  getStrapiMediaUrl,
} from "@/core/lib/api";
import { buildMetadataFromSeo } from "@/core/lib/seo";
import type { StrapiBlockNode } from "@/core/lib/api/types";
import { BlogSidebar, BlocksRenderer } from "@/modules/blog/components";
import { MarkdownRenderer } from "@/components/markdown-renderer";
import { Link } from "@/core/lib/i18n/navigation";
import { Calendar, Clock, ArrowLeft, Tag, User } from "lucide-react";

type Props = {
  params: Promise<{ locale: string; slug: string }>;
};

export async function generateMetadata({ params }: Props): Promise<Metadata> {
  const { locale, slug } = await params;
  const blog = await getBlogBySlug(slug, { locale });

  if (!blog) {
    return buildMetadataFromSeo({
      locale: (locale === "ar" ? "ar" : "en") as "en" | "ar",
      seo: {
        metaTitle: "Blog Post Not Found | YS Lootah Tech",
        metaDescription:
          "The requested article could not be found on YS Lootah Tech. Explore our latest insights on digital transformation, software development, AI, and cybersecurity.",
      },
    });
  }

  return buildMetadataFromSeo({
    seo: (blog as any).seo ?? undefined,
    openGraph: (blog as any).openGraph ?? undefined,
    locale: (locale === "ar" ? "ar" : "en") as "en" | "ar",
    override: {
      title: blog.title,
    },
  });
}

function getTextFromBlocks(blocks: StrapiBlockNode[]): string {
  return blocks.map(block => {
    if ('children' in block) {
      return block.children.map(c => 'text' in c ? c.text : '').join(' ');
    }
    return '';
  }).join(' ');
}

function estimateReadTime(content: string | StrapiBlockNode[] | null | undefined): string {
  if (!content) return "3 min read";

  let text = "";
  if (typeof content === "string") {
    text = content;
  } else if (Array.isArray(content)) {
    text = getTextFromBlocks(content);
  }

  const words = text.split(/\s+/).length;
  const minutes = Math.max(1, Math.ceil(words / 200));
  return `${minutes} min read`;
}

function formatDate(dateStr?: string | null): string {
  if (!dateStr) return "";
  return new Date(dateStr).toLocaleDateString("en-US", {
    year: "numeric",
    month: "long",
    day: "numeric",
  });
}

export default async function BlogDetailPage({ params }: Props) {
  const { locale, slug } = await params;
  setRequestLocale(locale);

  const [blog, tags, recentPosts] = await Promise.all([
    getBlogBySlug(slug, { locale }),
    getTags(), // Tags are usually shared or already localized by Strapi if i18n enabled on them
    getRecentBlogs(5, { locale }),
  ]);

  if (!blog) notFound();

  const coverUrl = getStrapiMediaUrl(blog.cover?.url);
  const readTime = estimateReadTime(blog.content);

  return (
    <div className="min-h-screen bg-slate-950">
      <Navbar />

      {/* Hero Cover */}
      <div className="relative h-[50vh] min-h-100 overflow-hidden">
        {coverUrl ? (
          <Image
            src={coverUrl}
            alt={blog.cover?.alternativeText || blog.title}
            fill
            priority
            className="object-cover"
            sizes="100vw"
          />
        ) : (
          <div className="h-full w-full bg-linear-to-br from-slate-800 to-slate-900" />
        )}

        {/* Overlays */}
        <div className="absolute inset-0 bg-linear-to-t from-slate-950 via-slate-950/60 to-slate-950/20" />
        <div className="absolute inset-0 bg-linear-to-b from-slate-950/80 via-transparent to-transparent" />

        {/* Content over hero */}
        <div className="absolute inset-0 flex items-end">
          <div className="container pb-10 md:pb-14">
            {/* Back button */}
            <Link
              href="/blog"
              className="group mb-6 inline-flex items-center gap-2 rounded-full border border-slate-700/50 bg-slate-900/50 px-4 py-2 text-sm text-slate-400 backdrop-blur-sm transition-colors hover:border-cyan-500/30 hover:text-cyan-400"
            >
              <ArrowLeft className="h-4 w-4 transition-transform group-hover:-translate-x-1" />
              Back to Blog
            </Link>

            {/* Category */}
            {blog.category?.name && (
              <div className="mb-4">
                <span className="rounded-full bg-cyan-500/20 px-3 py-1 text-xs font-medium text-cyan-400 border border-cyan-500/20">
                  {blog.category.name}
                </span>
              </div>
            )}

            {/* Title */}
            <h1 className="max-w-4xl text-3xl font-bold tracking-tight text-white md:text-4xl lg:text-5xl">
              {blog.title}
            </h1>

            {/* Meta row */}
            <div className="mt-6 flex flex-wrap items-center gap-5 text-sm text-slate-400">
              {blog.author?.username && (
                <span className="flex items-center gap-2">
                  <div className="flex h-8 w-8 items-center justify-center rounded-full bg-cyan-500/20 text-cyan-400">
                    <User className="h-4 w-4" />
                  </div>
                  <span className="font-medium text-slate-300">
                    {blog.author.username}
                  </span>
                </span>
              )}
              {blog.publishedAt && (
                <span className="flex items-center gap-1.5">
                  <Calendar className="h-4 w-4" />
                  {formatDate(blog.publishedAt)}
                </span>
              )}
              <span className="flex items-center gap-1.5">
                <Clock className="h-4 w-4" />
                {readTime}
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* Body */}
      <div className="container py-12 md:py-16">
        <div className="grid gap-10 lg:grid-cols-[1fr_320px]">
          {/* Article Content */}
          <article className="min-w-0">
            {/* Tags strip */}
            {blog.tags && blog.tags.length > 0 && (
              <div className="mb-8 flex flex-wrap gap-2">
                {blog.tags.map((tag) => (
                  <Link
                    key={tag.documentId}
                    href={`/blog?tag=${tag.slug}`}
                    className="inline-flex items-center gap-1.5 rounded-lg bg-slate-800/80 px-3 py-1.5 text-xs font-medium text-slate-400 transition-colors hover:bg-cyan-500/20 hover:text-cyan-400 border border-slate-700/50 hover:border-cyan-500/30"
                  >
                    <Tag className="h-3 w-3" />
                    {tag.title}
                  </Link>
                ))}
              </div>
            )}

            {/* Content Rendering: Markdown or Blocks */}
            {typeof blog.content === "string" ? (
              <MarkdownRenderer content={blog.content} />
            ) : Array.isArray(blog.content) ? (
              <BlocksRenderer content={blog.content} />
            ) : null}

            {/* Divider & back link */}
            <div className="mt-16 border-t border-slate-800/50 pt-8">
              <Link
                href="/blog"
                className="group inline-flex items-center gap-2 text-sm font-medium text-cyan-400 transition-colors hover:text-cyan-300"
              >
                <ArrowLeft className="h-4 w-4 transition-transform group-hover:-translate-x-1" />
                Back to all articles
              </Link>
            </div>
          </article>

          {/* Sidebar */}
          <BlogSidebar tags={tags} recentPosts={recentPosts} />
        </div>
      </div>

      <Footer />
    </div>
  );
}
